<?php
/**
 * Live FileMaker Integration mit eigener Session-ID
 * Ermöglicht FileMaker die Kontrolle über Session-IDs
 */

// Session starten
session_start();

// Header für UTF-8 Encoding
header("Content-Type: application/json; charset=UTF-8");

// CORS-Header für FileMaker
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Session-ID");

// Google Maps API Key - wird aus JSON-Request extrahiert
$apiKey = null;

// Debug-Modus
$debug = true;

// OPTIONS-Request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

/**
 * Bereinigt Eingabedaten
 */
function cleanData($data) {
    if (is_array($data)) {
        return array_map('cleanData', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Validiert FileMaker Session-ID
 */
function validateSessionId($sessionId) {
    // FileMaker Session-ID Format: FM_[timestamp]_[unique] oder custom format
    if (empty($sessionId)) {
        return false;
    }
    
    // Erlaubte Zeichen: Buchstaben, Zahlen, Unterstrich, Bindestrich, Punkt
    if (!preg_match('/^[a-zA-Z0-9_\-\.]+$/', $sessionId)) {
        return false;
    }
    
    // Länge prüfen (min 5, max 100 Zeichen)
    if (strlen($sessionId) < 5 || strlen($sessionId) > 100) {
        return false;
    }
    
    return true;
}

/**
 * Erstellt FileMaker-kompatible Antwort
 */
function createFileMakerResponse($success, $data = null, $error = null, $sessionId = null) {
    $response = [
        'success' => $success,
        'timestamp' => date('Y-m-d H:i:s'),
        'session_id' => $sessionId,
        'data' => $data,
        'error' => $error
    ];
    
    if ($sessionId) {
        $response['webviewer_url'] = generateWebViewerUrl($sessionId);
        $response['session_info_url'] = generateSessionInfoUrl($sessionId);
    }
    
    return $response;
}

/**
 * Generiert WebViewer-URL
 */
function generateWebViewerUrl($sessionId) {
    $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'];
    $path = dirname($_SERVER['PHP_SELF']);
    return $protocol . $host . $path . '/route_webviewer.php?session_id=' . urlencode($sessionId);
}

/**
 * Generiert Session-Info-URL
 */
function generateSessionInfoUrl($sessionId) {
    $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'];
    $path = dirname($_SERVER['PHP_SELF']);
    return $protocol . $host . $path . '/filemaker_live_integration.php?action=info&session_id=' . urlencode($sessionId);
}

/**
 * Lädt Session-Daten (aus Datei wenn PHP-Session leer)
 */
function loadSessionData($sessionId) {
    // Erst aus PHP-Session versuchen
    if (isset($_SESSION[$sessionId])) {
        $sessionData = $_SESSION[$sessionId];
        
        // Session-Ablauf prüfen
        if (time() > $sessionData['expires']) {
            unset($_SESSION[$sessionId]);
            // Auch Datei löschen
            $sessionFile = __DIR__ . '/sessions/' . $sessionId . '.json';
            if (file_exists($sessionFile)) {
                unlink($sessionFile);
            }
            return null;
        }
        
        return $sessionData['data'];
    }
    
    // Fallback: Aus Datei laden
    $sessionFile = __DIR__ . '/sessions/' . $sessionId . '.json';
    if (!file_exists($sessionFile)) {
        return null;
    }
    
    $sessionContent = file_get_contents($sessionFile);
    if ($sessionContent === false) {
        return null;
    }
    
    $sessionData = json_decode($sessionContent, true);
    if (!$sessionData || !isset($sessionData['expires'])) {
        return null;
    }
    
    // Session-Ablauf prüfen
    if (time() > $sessionData['expires']) {
        unlink($sessionFile);
        return null;
    }
    
    // Wieder in PHP-Session laden (für Performance)
    $_SESSION[$sessionId] = $sessionData;
    
    return $sessionData['data'];
}

/**
 * Speichert Session-Daten (persistent als Datei)
 */
function saveSessionData($sessionId, $data, $expirationHours = 24) {
    // Session-Verzeichnis erstellen falls nicht vorhanden
    $sessionDir = __DIR__ . '/sessions';
    if (!is_dir($sessionDir)) {
        mkdir($sessionDir, 0755, true);
    }
    
    $sessionFile = $sessionDir . '/' . $sessionId . '.json';
    $sessionWrapper = [
        'data' => $data,
        'timestamp' => time(),
        'expires' => time() + ($expirationHours * 60 * 60),
        'filemaker_managed' => true
    ];
    
    // Auch in PHP-Session speichern (für Kompatibilität)
    $_SESSION[$sessionId] = $sessionWrapper;
    
    // Als Datei speichern (persistent)
    $result = file_put_contents($sessionFile, json_encode($sessionWrapper), LOCK_EX);
    
    return $result !== false;
}

/**
 * Session-Logging für FileMaker
 */
function logFileMakerSession($sessionId, $action, $data = null, $result = null) {
    $logEntry = [
        'session_id' => $sessionId,
        'action' => $action,
        'timestamp' => date('Y-m-d H:i:s'),
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        'data_summary' => $data ? [
            'contact_count' => is_array($data) ? count($data) : 0,
            'has_options' => isset($data['options'])
        ] : null,
        'result_summary' => $result ? [
            'success' => $result['success'] ?? false,
            'error' => $result['error'] ?? null
        ] : null
    ];
    
    $logFile = __DIR__ . '/logs/filemaker_sessions.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
}


/**
 * Erkennt und korrigiert vertauschte Straße/Hausnummer-Felder
 */
function detectAndFixAddressFields($contact, $debug = false) {
    $originalStrasse = $contact['strasse'] ?? '';
    $originalNr = $contact['nr'] ?? '';
    
    if (!empty($originalStrasse) && !empty($originalNr)) {
        if (preg_match('/^[\d\/\-\s]+$/', $originalStrasse) && preg_match('/[a-zA-ZäöüÄÖÜß]/', $originalNr)) {
            if ($debug) {
                error_log("Adressfeld-Korrektur für {$contact['id_contact']}: '{$originalStrasse}' <-> '{$originalNr}'");
            }
            
            return [
                'strasse' => $originalNr,
                'nr' => $originalStrasse,
                'corrected' => true
            ];
        }
    }
    
    return [
        'strasse' => $originalStrasse,
        'nr' => $originalNr,
        'corrected' => false
    ];
}

/**
 * Geocodiert eine Adresse
 */
function geocodeAddress($street, $nr, $ort, $plz, $land, $apiKey) {
    $addressComponents = array_filter([
        $street . ' ' . $nr,
        $plz,
        $ort,
        $land
    ]);
    $address = implode(', ', $addressComponents);
    
    $url = "https://maps.googleapis.com/maps/api/geocode/json?address=" . urlencode($address) . "&key={$apiKey}";
    
    $response = file_get_contents($url);
    
    if ($response === false) {
        return [
            'success' => false,
            'error' => 'Fehler beim Abrufen der Geocoding API',
            'lat' => null,
            'lng' => null,
            'formatted_address' => null
        ];
    }
    
    $data = json_decode($response, true);
    
    if ($data['status'] === "OK" && isset($data['results'][0]['geometry']['location'])) {
        $location = $data['results'][0]['geometry']['location'];
        return [
            'success' => true,
            'error' => null,
            'lat' => $location['lat'],
            'lng' => $location['lng'],
            'formatted_address' => $data['results'][0]['formatted_address']
        ];
    } else {
        return [
            'success' => false,
            'error' => 'Keine Geodaten gefunden: ' . ($data['status'] ?? 'Unbekannter Fehler'),
            'lat' => null,
            'lng' => null,
            'formatted_address' => null
        ];
    }
}

/**
 * Verarbeitet einen Kontakt. Da wir wissen wie die Daten aus FileMaker kommen, ist es hier möglich eigene Werte zu definieren z.B. Notizen, Zeit vor ort ätc.
 */
function processContact($contact, $apiKey, $debug = false) {
    $addressFields = detectAndFixAddressFields($contact, $debug);
    
    $processedContact = [
        'id_contact' => cleanData($contact['id_contact'] ?? ''),
        'strasse' => cleanData($addressFields['strasse']),
        'nr' => cleanData($addressFields['nr']),
        'ort' => cleanData($contact['ort'] ?? ''),
        'plz' => cleanData($contact['plz'] ?? ''),
        'land' => cleanData($contact['land'] ?? 'Deutschland'),
        'first' => cleanData($contact['first'] ?? ''),
        'last' => cleanData($contact['last'] ?? ''),
        'full_name' => trim($contact['first'] . ' ' . $contact['last']),
        'full_address' => trim($addressFields['strasse'] . ' ' . $addressFields['nr'] . ', ' . $contact['plz'] . ' ' . $contact['ort'] . ', ' . ($contact['land'] ?? 'Deutschland')),
        'address_corrected' => $addressFields['corrected'],
        // FileMaker-spezifische Felder
        'filemaker_id' => cleanData($contact['filemaker_id'] ?? ''),
        'tour_date' => cleanData($contact['tour_date'] ?? ''),
        'tour_time' => cleanData($contact['tour_time'] ?? ''),
        'priority' => cleanData($contact['priority'] ?? ''),
        'notes' => cleanData($contact['notes'] ?? ''),
        'status' => cleanData($contact['status'] ?? 'Neu')
    ];
    
    // Geocoding durchführen
    $geocodeResult = geocodeAddress(
        $processedContact['strasse'], 
        $processedContact['nr'], 
        $processedContact['ort'], 
        $processedContact['plz'], 
        $processedContact['land'], 
        $apiKey
    );
    
    $processedContact['geocoding'] = $geocodeResult;
    
    return $processedContact;
}

/**
 * K-Means Clustering basierend auf GPS-Koordinaten
 */
function clusterByCoordinates($contacts, $maxClusterSize = 25) {
    $totalContacts = count($contacts);
    
    // Wenn weniger als maxClusterSize Kontakte, alle in einen Cluster
    if ($totalContacts <= $maxClusterSize) {
        return [$contacts];
    }
    
    // Berechne optimale Anzahl von Clustern
    $idealClusterCount = ceil($totalContacts / $maxClusterSize);
    
    // K-Means Clustering
    $clusters = kMeansCluster($contacts, $idealClusterCount, $maxClusterSize);
    
    return $clusters;
}

/**
 * K-Means Clustering Algorithmus
 */
function kMeansCluster($contacts, $k, $maxClusterSize = 25, $maxIterations = 100) {
    // Nur Kontakte mit gültigen Koordinaten verwenden
    $validContacts = array_filter($contacts, function($contact) {
        return isset($contact['geocoding']['success']) && 
               $contact['geocoding']['success'] && 
               isset($contact['geocoding']['lat']) && 
               isset($contact['geocoding']['lng']);
    });
    
    if (empty($validContacts)) {
        return [$contacts]; // Fallback: alle in einen Cluster
    }
    
    $n = count($validContacts);
    if ($n <= $k) {
        // Weniger Kontakte als gewünschte Cluster
        return array_map(function($contact) { return [$contact]; }, $validContacts);
    }
    
    // Initiale Zentroiden zufällig wählen
    $centroids = [];
    $usedIndices = [];
    for ($i = 0; $i < $k; $i++) {
        do {
            $randomIndex = rand(0, $n - 1);
        } while (in_array($randomIndex, $usedIndices));
        
        $usedIndices[] = $randomIndex;
        $contact = array_values($validContacts)[$randomIndex];
        $centroids[] = [
            'lat' => floatval($contact['geocoding']['lat']),
            'lng' => floatval($contact['geocoding']['lng'])
        ];
    }
    
    // K-Means Iterationen
    for ($iteration = 0; $iteration < $maxIterations; $iteration++) {
        // Kontakte zu nächstem Zentroid zuordnen
        $clusters = array_fill(0, $k, []);
        
        foreach ($validContacts as $contact) {
            $contactLat = floatval($contact['geocoding']['lat']);
            $contactLng = floatval($contact['geocoding']['lng']);
            
            $minDistance = PHP_FLOAT_MAX;
            $closestCluster = 0;
            
            for ($j = 0; $j < $k; $j++) {
                $distance = calculateDistance(
                    $contactLat, $contactLng,
                    $centroids[$j]['lat'], $centroids[$j]['lng']
                );
                
                if ($distance < $minDistance) {
                    $minDistance = $distance;
                    $closestCluster = $j;
                }
            }
            
            $clusters[$closestCluster][] = $contact;
        }
        
        // Neue Zentroiden berechnen
        $newCentroids = [];
        $centroidsChanged = false;
        
        for ($j = 0; $j < $k; $j++) {
            if (empty($clusters[$j])) {
                // Leerer Cluster - Zentroid beibehalten
                $newCentroids[] = $centroids[$j];
                continue;
            }
            
            $avgLat = 0;
            $avgLng = 0;
            foreach ($clusters[$j] as $contact) {
                $avgLat += floatval($contact['geocoding']['lat']);
                $avgLng += floatval($contact['geocoding']['lng']);
            }
            $avgLat /= count($clusters[$j]);
            $avgLng /= count($clusters[$j]);
            
            $newCentroid = ['lat' => $avgLat, 'lng' => $avgLng];
            $newCentroids[] = $newCentroid;
            
            // Prüfen ob sich Zentroid geändert hat
            if (abs($centroids[$j]['lat'] - $avgLat) > 0.001 || 
                abs($centroids[$j]['lng'] - $avgLng) > 0.001) {
                $centroidsChanged = true;
            }
        }
        
        $centroids = $newCentroids;
        
        // Konvergenz erreicht?
        if (!$centroidsChanged) {
            break;
        }
    }
    
    // Leere Cluster entfernen
    $clusters = array_filter($clusters, function($cluster) {
        return !empty($cluster);
    });
    
    // Zu große Cluster aufteilen
    $finalClusters = [];
    foreach ($clusters as $cluster) {
        if (count($cluster) <= $maxClusterSize) {
            $finalClusters[] = $cluster;
        } else {
            // Rekursiv aufteilen
            $subClusterCount = ceil(count($cluster) / $maxClusterSize);
            $subClusters = kMeansCluster($cluster, $subClusterCount, $maxClusterSize, 50);
            $finalClusters = array_merge($finalClusters, $subClusters);
        }
    }
    
    return array_values($finalClusters);
}

/**
 * PLZ-basiertes Clustering (Fallback) ist nur für den Notfall, da sehr ungenau sollten wir uns niemals auf PLZ verlassen.
 */
function clusterByPostalCode($contacts, $maxClusterSize = 25) {
    $totalContacts = count($contacts);
    
    // Wenn weniger als maxClusterSize Kontakte, alle in einen Cluster
    if ($totalContacts <= $maxClusterSize) {
        return [$contacts];
    }
    
    // Berechne optimale Anzahl von Clustern
    $idealClusterCount = ceil($totalContacts / $maxClusterSize);
    $idealClusterSize = ceil($totalContacts / $idealClusterCount);
    
    // Sortiere nach PLZ für geografische Nähe
    usort($contacts, function($a, $b) {
        return strcmp($a['plz'], $b['plz']);
    });
    
    $clusters = [];
    $currentCluster = [];
    
    foreach ($contacts as $contact) {
        $currentCluster[] = $contact;
        
        // Cluster schließen wenn:
        // 1. Maximalgröße erreicht ODER
        // 2. Ideale Größe erreicht und noch genug Kontakte für weitere Cluster übrig
        $remainingContacts = $totalContacts - array_sum(array_map('count', $clusters)) - count($currentCluster);
        $remainingClusters = $idealClusterCount - count($clusters) - 1;
        
        if (count($currentCluster) >= $maxClusterSize || 
            (count($currentCluster) >= $idealClusterSize && 
             $remainingClusters > 0 && 
             $remainingContacts >= $remainingClusters)) {
            
            $clusters[] = $currentCluster;
            $currentCluster = [];
        }
    }
    
    // Letzten Cluster hinzufügen falls nicht leer
    if (!empty($currentCluster)) {
        $clusters[] = $currentCluster;
    }
    
    return $clusters;
}

/**
 * Berechnet die Entfernung zwischen zwei GPS-Koordinaten in Kilometern
 * Verwendet die Haversine-Formel
 */
function calculateDistance($lat1, $lng1, $lat2, $lng2) {
    $earthRadius = 6371; // Erdradius in Kilometern
    
    $dLat = deg2rad($lat2 - $lat1);
    $dLng = deg2rad($lng2 - $lng1);
    
    $a = sin($dLat / 2) * sin($dLat / 2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLng / 2) * sin($dLng / 2);
    
    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    
    return $earthRadius * $c;
}

/**
 * Ermittelt die Straßenentfernung (in km) via Google Distance Matrix API
 * Fällt bei Fehlern auf null zurück (Aufrufer kann Haversine als Fallback nutzen)
 */
function distanceMatrixDistance($origLat, $origLng, $destLat, $destLng, $apiKey, $mode = 'driving', $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    $orig = $origLat . ',' . $origLng;
    $dest = $destLat . ',' . $destLng;
    $url = 'https://maps.googleapis.com/maps/api/distancematrix/json?units=metric'
        . '&mode=' . urlencode($mode)
        . '&origins=' . urlencode($orig)
        . '&destinations=' . urlencode($dest)
        . '&key=' . urlencode($apiKey);
    if (!empty($departureTimeEpoch)) {
        $url .= '&departure_time=' . urlencode((string)$departureTimeEpoch) . '&traffic_model=' . urlencode($trafficModel);
    }

    $response = @file_get_contents($url);
    if ($response === false) {
        return null;
    }

    $data = json_decode($response, true);
    if (!is_array($data) || ($data['status'] ?? '') !== 'OK') {
        return null;
    }

    $element = $data['rows'][0]['elements'][0] ?? null;
    if (!$element || ($element['status'] ?? '') !== 'OK') {
        return null;
    }

    // Distance in Metern -> Kilometer
    $meters = $element['distance']['value'] ?? null;
    if ($meters === null) {
        return null;
    }

    return $meters / 1000.0;
}

/**
 * Liefert Distance+Duration (optional in_traffic) via Distance Matrix
 */
function distanceMatrixMetrics($origLat, $origLng, $destLat, $destLng, $apiKey, $mode = 'driving', $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    $orig = $origLat . ',' . $origLng;
    $dest = $destLat . ',' . $destLng;
    $url = 'https://maps.googleapis.com/maps/api/distancematrix/json?units=metric'
        . '&mode=' . urlencode($mode)
        . '&origins=' . urlencode($orig)
        . '&destinations=' . urlencode($dest)
        . '&key=' . urlencode($apiKey);
    if (!empty($departureTimeEpoch)) {
        $url .= '&departure_time=' . urlencode((string)$departureTimeEpoch) . '&traffic_model=' . urlencode($trafficModel);
    }

    $response = @file_get_contents($url);
    if ($response === false) {
        return null;
    }

    $data = json_decode($response, true);
    if (!is_array($data) || ($data['status'] ?? '') !== 'OK') {
        return null;
    }

    $element = $data['rows'][0]['elements'][0] ?? null;
    if (!$element || ($element['status'] ?? '') !== 'OK') {
        return null;
    }

    $meters = $element['distance']['value'] ?? null;
    $durationS = $element['duration']['value'] ?? null;
    $durationInTrafficS = $element['duration_in_traffic']['value'] ?? null;
    if ($meters === null || $durationS === null) {
        return null;
    }

    return [
        'distance_km' => $meters / 1000.0,
        'duration_s' => (int)$durationS,
        'duration_in_traffic_s' => $durationInTrafficS !== null ? (int)$durationInTrafficS : null,
    ];
}

/**
 * Liefert Distanz in km – bevorzugt Distance Matrix, fallback Haversine
 */
function getDistanceKm($lat1, $lng1, $lat2, $lng2, $apiKey = null, $preferMatrix = true, $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    if ($preferMatrix && $apiKey) {
        $dm = distanceMatrixDistance($lat1, $lng1, $lat2, $lng2, $apiKey, 'driving', $departureTimeEpoch, $trafficModel);
        if ($dm !== null) {
            return $dm;
        }
    }
    return calculateDistance($lat1, $lng1, $lat2, $lng2);
}

/**
 * Normalisiert ein Kontaktpaar zu einem Cache-Key (reihenfolgeunabhängig)
 */
function makeDistanceCacheKey($lat1, $lng1, $lat2, $lng2, $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    $a = [round((float)$lat1, 6), round((float)$lng1, 6)];
    $b = [round((float)$lat2, 6), round((float)$lng2, 6)];
    $pair = [$a, $b];
    sort($pair);
    $meta = [
        'departure' => $departureTimeEpoch ? (int)floor($departureTimeEpoch / 60) : null, // minuten-genau cachen
        'traffic' => $trafficModel,
    ];
    return md5(json_encode([$pair, $meta]));
}

/**
 * Distanz mit einfachem In-Memory-Cache
 */
function getDistanceKmCached($lat1, $lng1, $lat2, $lng2, $apiKey, &$cache, $preferMatrix = true, $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    $key = makeDistanceCacheKey($lat1, $lng1, $lat2, $lng2, $departureTimeEpoch, $trafficModel);
    if (isset($cache[$key])) {
        return $cache[$key];
    }
    $distance = getDistanceKm($lat1, $lng1, $lat2, $lng2, $apiKey, $preferMatrix, $departureTimeEpoch, $trafficModel);
    $cache[$key] = $distance;
    return $distance;
}

/**
 * Metrics (Distanz + Dauer) mit Cache
 */
function getTravelMetricsCached($lat1, $lng1, $lat2, $lng2, $apiKey, &$cache, $preferMatrix = true, $departureTimeEpoch = null, $trafficModel = 'best_guess') {
    $key = 'metrics:' . makeDistanceCacheKey($lat1, $lng1, $lat2, $lng2, $departureTimeEpoch, $trafficModel);
    if (isset($cache[$key])) {
        return $cache[$key];
    }
    if ($preferMatrix && $apiKey) {
        $metrics = distanceMatrixMetrics($lat1, $lng1, $lat2, $lng2, $apiKey, 'driving', $departureTimeEpoch, $trafficModel);
        if ($metrics !== null) {
            $cache[$key] = $metrics;
            return $metrics;
        }
    }
    // Fallback nur Distanz, Dauer unbekannt
    $distance = calculateDistance($lat1, $lng1, $lat2, $lng2);
    $result = [
        'distance_km' => $distance,
        'duration_s' => null,
        'duration_in_traffic_s' => null,
    ];
    $cache[$key] = $result;
    return $result;
}

/**
 * Generiert Cluster-Farben für die Darstellung. Dies ist begrenzt auf 20 Farben. Kann allerdings noch erweitert werden
 */
function getClusterColor($clusterIndex) {
    $colors = [
        '#FF6B6B', // Rot
        '#4ECDC4', // Türkis
        '#45B7D1', // Blau
        '#96CEB4', // Grün
        '#FECA57', // Gelb
        '#FF9FF3', // Pink
        '#54A0FF', // Hellblau
        '#5F27CD', // Lila
        '#00D2D3', // Cyan
        '#FF9F43', // Orange
        '#6C5CE7', // Violett
        '#A3CB38', // Limette
        '#FD79A8', // Rosa
        '#E17055', // Terrakotta
        '#74B9FF', // Himmelblau
        '#81ECEC', // Aqua
        '#FAB1A0', // Pfirsich
        '#00B894', // Smaragd
        '#FDCB6E', // Honig
        '#6C5CE7'  // Indigo
    ];
    
    return $colors[$clusterIndex % count($colors)];
}

// Hauptverarbeitung
try {
    $method = $_SERVER['REQUEST_METHOD'];
    
    if ($method === 'POST') {
        // POST: Neue Session erstellen oder bestehende aktualisieren
        $input = file_get_contents('php://input');
        
        if (empty($input)) {
            echo json_encode(createFileMakerResponse(false, null, 'Keine POST-Daten empfangen'));
            exit;
        }
        
        $requestData = json_decode($input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            echo json_encode(createFileMakerResponse(false, null, 'Ungültiges JSON: ' . json_last_error_msg()));
            exit;
        }
        
        // FileMaker Session-ID aus Request extrahieren
        $sessionId = $requestData['session_id'] ?? $_SERVER['HTTP_X_SESSION_ID'] ?? null;
        
        if (!$sessionId) {
            echo json_encode(createFileMakerResponse(false, null, 'Session-ID fehlt. Bitte session_id im JSON oder X-Session-ID Header übertragen.'));
            exit;
        }
        
        if (!validateSessionId($sessionId)) {
            echo json_encode(createFileMakerResponse(false, null, 'Ungültige Session-ID. Erlaubt: Buchstaben, Zahlen, Unterstrich, Bindestrich (5-100 Zeichen)'));
            exit;
        }
        
        // Kontakte und Parameter extrahieren (FileMaker-Format)
        $contacts = $requestData['data'] ?? $requestData['contacts'] ?? [];
        $options = $requestData['options'] ?? [];
        
        // Google API Key aus JSON-Request extrahieren
        $apiKey = $requestData['google_key'] ?? null;
        if (!$apiKey) {
            echo json_encode(createFileMakerResponse(false, null, 'Google API Key (google_key) fehlt im JSON-Request', $sessionId));
            exit;
        }
        
        // max_cluster_size direkt aus JSON-Request extrahieren
        $maxClusterSize = $requestData['max_cluster_size'] ?? $options['max_cluster_size'] ?? 25;
        
        // Startzeit (p_zeit) optional entgegennehmen (ISO, Datum+Zeit oder Epoch)
        $pZeitRaw = $requestData['p_zeit'] ?? $options['p_zeit'] ?? null;
        $departureTimeEpoch = null;
        if (!empty($pZeitRaw)) {
            if ($debug) {
                error_log("DEBUG p_zeit: '$pZeitRaw' (type: " . gettype($pZeitRaw) . ")");
            }
            
            if (is_numeric($pZeitRaw)) {
                $departureTimeEpoch = (int)$pZeitRaw;
                if ($debug) {
                    error_log("DEBUG: Numeric p_zeit -> epoch: $departureTimeEpoch");
                }
            } else {
                // Erst prüfen ob es nur eine Zeit ohne Datum ist (z.B. "08:30" oder "08:30:00")
                if (preg_match('/^\d{1,2}:\d{2}(:\d{2})?$/', trim($pZeitRaw))) {
                    $today = date('Y-m-d');
                    $timeOnly = trim($pZeitRaw);
                    $fullDateTime = $today . ' ' . $timeOnly;
                    $ts = strtotime($fullDateTime);
                    
                    if ($debug) {
                        error_log("DEBUG: Time-only detected. Today: $today, Full: $fullDateTime, TS: $ts");
                    }
                    
                    // Falls die Zeit heute bereits vorbei ist, auf morgen setzen
                    if ($ts !== false && $ts < time()) {
                        $ts = strtotime('+1 day', $ts);
                        if ($debug) {
                            error_log("DEBUG: Time in past, moved to tomorrow. New TS: $ts");
                        }
                    }
                    $departureTimeEpoch = $ts;
                } else {
                    // Normale strtotime-Verarbeitung für andere Formate
                    $ts = strtotime($pZeitRaw);
                    if ($ts !== false) {
                        $departureTimeEpoch = $ts;
                        if ($debug) {
                            error_log("DEBUG: Normal strtotime -> epoch: $departureTimeEpoch");
                        }
                    }
                }
            }
        }
        
        if ($debug && $departureTimeEpoch) {
            error_log("DEBUG: Final departure_time_epoch: $departureTimeEpoch (" . date('Y-m-d H:i:s', $departureTimeEpoch) . ")");
        }

        // Standard-Optionen
        $clusterMethod = $options['cluster_method'] ?? 'k_means_coordinates';
        $tourDate = $options['tour_date'] ?? date('Y-m-d');
        $tourName = $options['tour_name'] ?? 'FileMaker Route ' . date('Y-m-d H:i');
        
        if (empty($contacts)) {
            echo json_encode(createFileMakerResponse(false, null, 'Keine Kontakte empfangen', $sessionId));
            exit;
        }
        
        if ($debug) {
            error_log("FileMaker Live-Integration: Session $sessionId mit " . count($contacts) . " Kontakten");
        }
        
        // Kontakte verarbeiten
        $processedContacts = [];
        $successCount = 0;
        $errorCount = 0;
        
        foreach ($contacts as $contact) {
            $processedContact = processContact($contact, $apiKey, $debug);
            $processedContacts[] = $processedContact;
            
            if ($processedContact['geocoding']['success']) {
                $successCount++;
            } else {
                $errorCount++;
            }
        }
        
        // Clustering durchführen (K-Means basierend auf Koordinaten)
        $clusters = clusterByCoordinates($processedContacts, $maxClusterSize);
        
        // Session-Daten erstellen
        $sessionData = [
            'session_id' => $sessionId,
            'tour_name' => $tourName,
            'tour_date' => $tourDate,
            'contacts' => $processedContacts,
            'clusters' => $clusters,
            'cluster_method' => $clusterMethod,
            'max_cluster_size' => $maxClusterSize,
            'total_contacts' => count($processedContacts),
            'total_clusters' => count($clusters),
            'api_key' => $apiKey,
            'departure_time_epoch' => $departureTimeEpoch,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'filemaker_options' => $options,
            'geocoding_stats' => [
                'successful' => $successCount,
                'failed' => $errorCount,
                'success_rate' => count($processedContacts) > 0 ? 
                    round(($successCount / count($processedContacts)) * 100, 2) : 0
            ]
        ];
        
        // Session speichern
        saveSessionData($sessionId, $sessionData);
        
        // Logging
        logFileMakerSession($sessionId, 'create_update', $requestData, ['success' => true]);
        
        // Antwort erstellen
        $responseData = [
            'session_id' => $sessionId,
            'tour_name' => $tourName,
            'tour_date' => $tourDate,
            'total_contacts' => count($processedContacts),
            'successful_geocoding' => $successCount,
            'failed_geocoding' => $errorCount,
            'success_rate' => $sessionData['geocoding_stats']['success_rate'],
            'total_clusters' => count($clusters),
            'cluster_sizes' => array_map('count', $clusters),
            'cluster_method' => $clusterMethod,
            'ready_for_optimization' => true
        ];
        
        echo json_encode(createFileMakerResponse(true, $responseData, null, $sessionId));
        
    } elseif ($method === 'GET') {
        // GET: Session-Info, Status oder Management
        $action = $_GET['action'] ?? 'info';
        $sessionId = $_GET['session_id'] ?? '';
        
        if (empty($sessionId)) {
            echo json_encode(createFileMakerResponse(false, null, 'Session-ID fehlt'));
            exit;
        }
        
        if (!validateSessionId($sessionId)) {
            echo json_encode(createFileMakerResponse(false, null, 'Ungültige Session-ID'));
            exit;
        }
        
        switch ($action) {
            case 'export':
                // Export: Vereinfachte Route-Liste für FileMaker - nur wesentliche Felder
                $sessionData = loadSessionData($sessionId);
                
                if (!$sessionData) {
                    echo json_encode(createFileMakerResponse(false, null, 'Session nicht gefunden oder abgelaufen', $sessionId));
                    exit;
                }
                
                // Route-Export erstellen - gewünschte Felder mit Kilometerdistanzen und -dauer
                $routeExport = [];
                $distanceCache = [];
                $departureTimeEpoch = $sessionData['departure_time_epoch'] ?? null;
                $trafficModel = 'best_guess';
                
                foreach ($sessionData['clusters'] as $clusterIndex => $cluster) {
                    $clusterColor = getClusterColor($clusterIndex);
                    
                    foreach ($cluster as $contactIndex => $contact) {
                        $routeEntry = [
                            'contact_id' => $contact['id_contact'],
                            'cluster_nr' => $clusterIndex + 1,
                            'route_order' => $contactIndex + 1,
                            'cluster_color' => $clusterColor,
                            'distance_to_next_km' => null,
                            'duration_to_next_s' => null,
                            'distance_to_previous_km' => null,
                            'duration_to_previous_s' => null
                        ];
                        
                        // Entfernung zum nächsten Kontakt im Cluster berechnen
                        if ($contactIndex < count($cluster) - 1) {
                            $nextContact = $cluster[$contactIndex + 1];
                            
                            // Prüfen ob beide Kontakte gültige GPS-Koordinaten haben
                            if (isset($contact['geocoding']['success']) && $contact['geocoding']['success'] &&
                                isset($nextContact['geocoding']['success']) && $nextContact['geocoding']['success']) {
                                
                                $metrics = getTravelMetricsCached(
                                    $contact['geocoding']['lat'],
                                    $contact['geocoding']['lng'],
                                    $nextContact['geocoding']['lat'],
                                    $nextContact['geocoding']['lng'],
                                    $sessionData['api_key'] ?? null,
                                    $distanceCache,
                                    true,
                                    $departureTimeEpoch,
                                    $trafficModel
                                );
                                if ($metrics) {
                                    $routeEntry['distance_to_next_km'] = isset($metrics['distance_km']) ? round($metrics['distance_km'], 2) : null;
                                    $routeEntry['duration_to_next_s'] = $metrics['duration_in_traffic_s'] ?? $metrics['duration_s'] ?? null;
                                }
                            }
                        }
                        
                        // Entfernung zum vorherigen Kontakt im Cluster berechnen
                        if ($contactIndex > 0) {
                            $previousContact = $cluster[$contactIndex - 1];
                            
                            // Prüfen ob beide Kontakte gültige GPS-Koordinaten haben
                            if (isset($contact['geocoding']['success']) && $contact['geocoding']['success'] &&
                                isset($previousContact['geocoding']['success']) && $previousContact['geocoding']['success']) {
                                
                                $metricsPrev = getTravelMetricsCached(
                                    $previousContact['geocoding']['lat'],
                                    $previousContact['geocoding']['lng'],
                                    $contact['geocoding']['lat'],
                                    $contact['geocoding']['lng'],
                                    $sessionData['api_key'] ?? null,
                                    $distanceCache,
                                    true,
                                    $departureTimeEpoch,
                                    $trafficModel
                                );
                                if ($metricsPrev) {
                                    $routeEntry['distance_to_previous_km'] = isset($metricsPrev['distance_km']) ? round($metricsPrev['distance_km'], 2) : null;
                                    $routeEntry['duration_to_previous_s'] = $metricsPrev['duration_in_traffic_s'] ?? $metricsPrev['duration_s'] ?? null;
                                }
                            }
                        }
                        
                        $routeExport[] = $routeEntry;
                    }
                }
                
                echo json_encode(createFileMakerResponse(true, $routeExport, null, $sessionId));
                break;
                
            case 'info':
                $sessionData = loadSessionData($sessionId);
                
                if ($sessionData === null) {
                    echo json_encode(createFileMakerResponse(false, null, 'Session nicht gefunden oder abgelaufen', $sessionId));
                } else {
                    $responseData = [
                        'session_id' => $sessionId,
                        'tour_name' => $sessionData['tour_name'] ?? 'Unbenannte Tour',
                        'tour_date' => $sessionData['tour_date'] ?? date('Y-m-d'),
                        'created_at' => $sessionData['created_at'],
                        'updated_at' => $sessionData['updated_at'] ?? $sessionData['created_at'],
                        'total_contacts' => $sessionData['total_contacts'],
                        'total_clusters' => $sessionData['total_clusters'],
                        'geocoding_stats' => $sessionData['geocoding_stats'],
                        'cluster_method' => $sessionData['cluster_method'],
                        'expires_at' => date('Y-m-d H:i:s', $_SESSION[$sessionId]['expires'])
                    ];
                    
                    echo json_encode(createFileMakerResponse(true, $responseData, null, $sessionId));
                }
                break;
                
            case 'delete':
                if (isset($_SESSION[$sessionId])) {
                    unset($_SESSION[$sessionId]);
                    logFileMakerSession($sessionId, 'delete');
                    echo json_encode(createFileMakerResponse(true, ['deleted' => true], null, $sessionId));
                } else {
                    echo json_encode(createFileMakerResponse(false, null, 'Session nicht gefunden', $sessionId));
                }
                break;
                
            case 'extend':
                if (isset($_SESSION[$sessionId])) {
                    $_SESSION[$sessionId]['expires'] = time() + (24 * 60 * 60);
                    logFileMakerSession($sessionId, 'extend');
                    echo json_encode(createFileMakerResponse(true, ['extended' => true, 'new_expiry' => date('Y-m-d H:i:s', $_SESSION[$sessionId]['expires'])], null, $sessionId));
                } else {
                    echo json_encode(createFileMakerResponse(false, null, 'Session nicht gefunden', $sessionId));
                }
                break;
                
            default:
                echo json_encode(createFileMakerResponse(false, null, 'Unbekannte Aktion: ' . $action, $sessionId));
        }
        
    } else {
        http_response_code(405);
        echo json_encode(createFileMakerResponse(false, null, 'Methode nicht erlaubt'));
    }
    
} catch (Exception $e) {
    http_response_code(500);
    $response = createFileMakerResponse(false, null, 'Server-Fehler: ' . $e->getMessage());
    echo json_encode($response);
    
    if ($debug) {
        error_log("FileMaker Live-Integration Fehler: " . $e->getMessage());
    }
}
?>
